using System;
using System.IO;
using System.Runtime.InteropServices;
using DynaPDF;

namespace text_search
{
	class TextSearch
	{
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      // This class marks the search text in the PDF file.
      private static CTextSearch m_TextSearch;

      static int  parseBeginTemplate(IntPtr Data, IntPtr Object, int Handle, ref TPDFRect BBox, IntPtr Matrix)
      {
         return m_TextSearch.BeginTemplate(BBox, Matrix);
      }

      static void parseEndTemplate(IntPtr Data)
      {
         m_TextSearch.EndTemplate();
      }

      static void parseMulMatrix(IntPtr Data, IntPtr Object, ref TCTM Matrix)
      {
         m_TextSearch.MulMatrix(Matrix);
      }

      static int parseRestoreGraphicState(IntPtr Data)
      {
         m_TextSearch.RestoreGState();
         return 0;
      }

      static int parseSaveGraphicState(IntPtr Data)
      {
         return m_TextSearch.SaveGState();
      }

      static void parseSetCharSpacing(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextSearch.SetCharSpacing(Value);
      }

      static void parseSetFont(IntPtr Data, IntPtr Object, TFontType Type, int Embedded, IntPtr FontName, TFStyle Style, double FontSize, IntPtr IFont)
      {
         m_TextSearch.SetFont(FontSize, Type, IFont);
      }

      static void parseSetTextDrawMode(IntPtr Data, IntPtr Object, TDrawMode Mode)
      {
         m_TextSearch.SetTextDrawMode(Mode);
      }

      static void parseSetTextScale(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextSearch.SetTextScale(Value);
      }

      static void parseSetWordSpacing(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextSearch.SetWordSpacing(Value);
      }

      static int parseShowTextArrayA(IntPtr Data, IntPtr Object, ref TCTM Matrix, TTextRecordA[] Source, int Count, double Width)
      {
         return m_TextSearch.MarkText(Matrix, Source, Count, Width);
      }

		[STAThread]
		static void Main(string[] args)
		{
		   int selCount = 0;
		   try
		   {
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // The output file wil be created later

            // External CMaps should always be loaded when processing text from PDF files.
            // See the description of ParseContent() for further information.
            pdf.SetCMapDir(System.IO.Path.GetFullPath("../../../../../Resource/CMap/"), DynaPDF.TLoadCMapFlags.lcmDelayed | DynaPDF.TLoadCMapFlags.lcmRecursive);

            // We avoid the conversion of pages to templates
            pdf.SetImportFlags(TImportFlags.ifImportAll | TImportFlags.ifImportAsPage);
            if (pdf.OpenImportFile("../../../../../dynapdf_help.pdf", TPwdType.ptOpen, null) < 0)
            {
               Console.Write("Input file \"../../../../../dynapdf_help.pdf\" not found!\n");
               Console.Read();
               return;
            }
            pdf.ImportPDFFile(1, 1.0, 1.0);
            pdf.CloseImportFile();

            // We flatten markup annotations and form fields so that we can search the text in these objects too.
            pdf.FlattenAnnots(TAnnotFlattenFlags.affMarkupAnnots);
            pdf.FlattenForm();

            TPDFParseInterface  stack = new TPDFParseInterface();
            stack.BeginTemplate       = new TBeginTemplate(parseBeginTemplate);
            stack.EndTemplate         = new TEndTemplate(parseEndTemplate);
            stack.MulMatrix           = new TMulMatrix(parseMulMatrix);
            stack.RestoreGraphicState = new TRestoreGraphicState(parseRestoreGraphicState);
            stack.SaveGraphicState    = new TSaveGraphicState(parseSaveGraphicState);
            stack.SetCharSpacing      = new TSetCharSpacing(parseSetCharSpacing);
            stack.SetFont             = new TSetFont(parseSetFont);
            stack.SetTextDrawMode     = new TSetTextDrawMode(parseSetTextDrawMode);
            stack.SetTextScale        = new TSetTextScale(parseSetTextScale);
            stack.SetWordSpacing      = new TSetWordSpacing(parseSetWordSpacing);
            stack.ShowTextArrayA      = new TShowTextArrayA(parseShowTextArrayA);

            // We draw rectangles on the position where the search string was found. To make the text
            // in background visible we use the blend mode bmMultiply. Adobes Acrobat rasters a page
            // without anti-aliasing when a blend mode is used. Don't wonder that the rasterizing
            // quality is worse in comparison to normal PDF files.
            TPDFExtGState g = new TPDFExtGState();
            pdf.InitExtGState(ref g);
            g.BlendMode = TBlendMode.bmMultiply;
            int gs = pdf.CreateExtGState(ref g);

            // We try to find this string in the PDF file
            m_TextSearch = new CTextSearch(pdf);
            m_TextSearch.SetSearchText("PDF");
            int i, count = pdf.GetPageCount();
            for (i = 1; i <= count; i++)
            {
               pdf.EditPage(i);

               pdf.SetExtGState(gs);
               pdf.SetFillColor(CPDF.PDF_YELLOW);

               m_TextSearch.Init();
               pdf.ParseContent(IntPtr.Zero, ref stack, DynaPDF.TParseFlags.pfNone);
               pdf.EndPage();
               if (m_TextSearch.GetSelCount() > 0)
               {
                  selCount += m_TextSearch.GetSelCount();
                  Console.Write("Found string on Page: {0} {1} times!\n", i, m_TextSearch.GetSelCount());
               }
            }
            if (pdf.HaveOpenDoc())
            {
               // We write the output file into the current directory.
               String filePath = Directory.GetCurrentDirectory() + "\\out.pdf";
               // OK, now we can open the output file.
               if (!pdf.OpenOutputFile(filePath)) return;
               if (pdf.CloseFile())
               {
                  System.Diagnostics.Process p = new System.Diagnostics.Process();
                  p.StartInfo.FileName = filePath;
                  p.Start();
               }
            }
            pdf = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
         }
         Console.Write("\nFound string in the file {0} times!\n", selCount);
         Console.Read();
		}
	}
}
